/*

Copyright 2008 ToastFreeware (Philipp Spitzer, Gregor Herrmann)

This file is part of Pocketlamp.

Pocketlamp is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation, either version 3 of the License, or (at your option)
any later version.

Pocketlamp is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
more details.

You should have received a copy of the GNU General Public License along with
Pocketlamp. If not, see <http://www.gnu.org/licenses/>.

*/

package at.priv.toastfreeware.pocketlamp;

import java.util.Map;

import android.app.NotificationManager;
import android.content.Context;
import android.graphics.Canvas;
import android.os.Handler;
import android.os.Message;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.View;


public class PocketLampView extends View {
	public enum Mode {STATIC, INTERACTIVE, BLINK};

	private int fgColor;          // current foreground color
	private int bgColor;          // current background color
	private Mode mode;            // current mode
	private double blinkDuration;   // duration of blinking in ms.
	private final double blinkChangeFactor = 1.5;
	private final double blinkChangeDelta = 100;
	private boolean showsBgColor;
	private boolean keyIsPressed;
	private NotificationManager notificationManager;
	private WakeLock wakeLock;

	private RefreshHandler redrawHandler = new RefreshHandler();

	class RefreshHandler extends Handler {

		@Override
		public void handleMessage(Message msg) {
			// one blinkDuration is over: blink!
			if (mode == Mode.BLINK) {
				showsBgColor = !showsBgColor;
				PocketLampView.this.invalidate();
				// make sure we are called again
				sleep(blinkDuration);
			}
		}


		public void sleep(double delayMillis) {
			this.removeMessages(0);
			sendMessageDelayed(obtainMessage(0), (long) delayMillis);
		}
	};


	@SuppressWarnings("unchecked")
	public PocketLampView(Context context, AttributeSet attrs, Map inflateParams) {
		super(context, attrs, inflateParams);
		setFocusable(true);
		keyIsPressed = false;
		reset();
		notificationManager = (NotificationManager) getContext().getSystemService(Context.NOTIFICATION_SERVICE);
		notificationManager.notifyWithText(R.string.notification_startup, getContext().getText(R.string.notification_startup), NotificationManager.LENGTH_SHORT, null);
		PowerManager powerManager = (PowerManager) mContext.getSystemService(Context.POWER_SERVICE);
		wakeLock = powerManager.newWakeLock(PowerManager.FULL_WAKE_LOCK, "PocketLampView");
		wakeLock.acquire();
	}


	@Override
	protected void finalize() throws Throwable {
		wakeLock.release();
		super.finalize();
	}


	public void reset() {
		fgColor = getResources().getColor(R.color.defaultFgColor);
		bgColor = getResources().getColor(R.color.defaultBgColor);
		mode = Mode.STATIC;
		blinkDuration = 500;
		showsBgColor = false;
		invalidate();
	}


	@Override
	public void onDraw(Canvas canvas) {
		super.onDraw(canvas);
		switch (mode) {
			case STATIC: 
				canvas.drawColor(fgColor);
				break;
			case INTERACTIVE:
				canvas.drawColor(keyIsPressed ? fgColor : bgColor);
				break;
			case BLINK:
				canvas.drawColor(showsBgColor ? bgColor : fgColor);
				break;
		}
	}


	private boolean isInteractiveKey(int keyCode) {
		switch (keyCode) {
			case KeyEvent.KEYCODE_5:
			case KeyEvent.KEYCODE_DPAD_CENTER: 
			return true;
		}
		return false;
	}


	@Override
	public boolean onKeyDown(int keyCode, KeyEvent msg) {
		if (isInteractiveKey(keyCode)) {
			keyIsPressed = true;
			if (mode == Mode.INTERACTIVE) {
				invalidate();
				return true;
			}
		}
		if (mode == Mode.BLINK) {
			if (keyCode == KeyEvent.KEYCODE_DPAD_UP) {
				blinkDuration /= blinkChangeFactor;
				return true;
			}
			if (keyCode == KeyEvent.KEYCODE_DPAD_DOWN) {
				blinkDuration *= blinkChangeFactor;
				return true;
			}
			if (keyCode == KeyEvent.KEYCODE_DPAD_RIGHT) {
				if (blinkDuration - blinkChangeDelta > 0) blinkDuration -= blinkChangeDelta;
				return true;
			}
			if (keyCode == KeyEvent.KEYCODE_DPAD_LEFT) {
				blinkDuration += blinkChangeDelta;
				return true;
			}
		}
		return false;
	}


	@Override
	public boolean onKeyUp(int keyCode, KeyEvent msg) {
		if (isInteractiveKey(keyCode)) {
			keyIsPressed = false;
			if (mode == Mode.INTERACTIVE) { 
				invalidate();
				return true;
			}
		}
		return false;
	}


	public int getFgColor() {
		return fgColor;
	}


	public void setFgColor(int color) {
		this.fgColor = color;
		invalidate();
	}


	public Mode getMode() {
		return mode;
	}


	public void setMode(Mode mode) {
		if (mode == Mode.INTERACTIVE)
			notificationManager.notifyWithText(R.string.notification_interactive, getContext().getText(R.string.notification_interactive), NotificationManager.LENGTH_SHORT, null);
		if (mode == this.mode) return;
		this.mode = mode;
		switch (mode) {
			case STATIC:
				showsBgColor = false;
				break;
			case INTERACTIVE: break;
			case BLINK:
				showsBgColor = false;
				notificationManager.notifyWithText(R.string.notification_blink, getContext().getText(R.string.notification_blink), NotificationManager.LENGTH_SHORT, null);
				redrawHandler.sleep(blinkDuration);
				break;
		}
		invalidate();
	}


	public double getBlinkDuration() {
		return blinkDuration;
	}


	public void setBlinkDuration(double blinkDuration) {
		this.blinkDuration = blinkDuration;
	}

}
