/*
# Copyright and License:
#
# Copyright (C) 2007-2009
# gregor herrmann <gregor+debian@comodo.priv.at>,
# Philipp Spitzer <philipp@spitzer.priv.at>
#
# This program is free software; you can redistribute it and/or modify it   
# under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA, or point
# your browser to http://www.gnu.org/licenses/gpl.html
*/

#include <QtGlobal>
#include <QtGui>

#include <iostream> // for help output
#include <string>   // for help output
#include <sys/stat.h> // glibc for umask and mkfifo
#include <errno.h>

#include "main.h"
#include "options.h"


// Types
// =====

VlcPlayer::VlcPlayer() {
	name = "vlc";
}


bool VlcPlayer::play(QString urlToPlay, QTime offset, bool saveOnly, QString saveFileName, QString& errorMsg) {
	if (saveOnly && saveFileName.isEmpty()) return true;
	QStringList arguments;
	arguments.append(urlToPlay);
	arguments.append("--start-time");
	arguments.append(QString::number(QTime(0, 0, 0, 0).secsTo(offset)));
	if (!saveFileName.isEmpty()) {
		arguments.append("--sout");
		arguments.append(QString("#transcode{vcodec=mp4v,vb=384,scale=1,acodec=mpga,ab=128,channels=2}:duplicate{%1dst=std{access=file,mux=mp4,dst=\"%2\"}}").arg(saveOnly ? "" : "dst=display,").arg(saveFileName));
	}
	QProcess player(0);
	qDebug() << "vlc" << arguments;
	if (!player.startDetached("vlc", arguments)) {
		errorMsg = QObject::tr("Could not start vlc.");
		return false;
	}
	return true;
}


MPlayer::MPlayer() {
	name = "mplayer";
	download = 0;
	#ifndef QT_4_1_COMPATIBLE
	tee = 0;
	player = 0;
	#else
	shell = 0;
	#endif
	tempFile = QDir::tempPath() + "teleschorschfifo." + qApp->sessionId();
}


MPlayer::~MPlayer() {
	stopProcesses();
	if (QFile::exists(tempFile)) QFile::remove(tempFile);
}


void MPlayer::stopProcesses() {
	if (download) {download->terminate(); download = 0;}
	#ifndef QT_4_1_COMPATIBLE
	if (player) {player->terminate(); player = 0;}
	if (tee) {tee->terminate(); tee = 0;}
	#else
	if (shell) {shell->terminate(); shell = 0;}
	#endif
	// If the QProcesses are deleted, a relatively long timeout occurs
	/*
	delete player;
	delete tee;
	delete download;
	delete shell;
	*/

}


// Play & save with mplayer in the bash:
// mkfifo zibpipe
// mplayer -dumpstream -dumpfile zibpipe "rtsp://81.189.213.1:554/orf2/zb/zib070423l.rm?URL=/ramgen/orf2/zb/zib070423l.rm&cloakport=8088,554,7070" >/dev/null 2>&1 | tee pipeout < zibpipe | mplayer -cache 512 -
bool MPlayer::play(QString urlToPlay, QTime offset, bool saveOnly, QString saveFileName, QString& errorMsg) {
	stopProcesses();
	if (saveOnly && saveFileName.isEmpty()) return true;
	QByteArray pipeFileName = tempFile.toLocal8Bit();
	static const char* pipeFile = pipeFileName.data();
	static const QString notStartError = QObject::tr("Could not start %1.");
	QStringList arguments;
	arguments.append(urlToPlay);
	arguments.append("-prefer-ipv4");    // -"- avoids IPV6 error message
	arguments.append("-ss"); 
	arguments.append(QString::number(QTime(0, 0, 0, 0).secsTo(offset)));
	if (!saveFileName.isEmpty()) {
		arguments.append("-dumpstream");
		arguments.append("-dumpfile");
		if (saveOnly) {
			arguments.append(saveFileName);
		} else {
			// Create fifo file ("named pipe") if it does not exist
			if (!QFile::exists(pipeFile)) {
				int result = mkfifo(pipeFile, 0664);
				if (result) {
					switch (errno) {
						case EACCES: errorMsg = "EACCES"; break;
						case ENAMETOOLONG: errorMsg = "ENAMETOOLONG"; break;
						case ENOENT: errorMsg = "ENOENT"; break;
						case ENOTDIR: errorMsg = "ENOTDIR"; break;
						case ELOOP: errorMsg = "ELOOP"; break;
						case EEXIST: errorMsg = "EACCESS"; break;
						case ENOSPC: errorMsg = "ENOSPC"; break;
						case EROFS: errorMsg = "EROFS"; break;
						default: errorMsg = QObject::tr("unknown error code");
					}
					errorMsg = QObject::tr("The pipe %1 cannot be created (%2)").arg(pipeFile).arg(errorMsg);
					return false;
				}
			}
			arguments.append(pipeFile);
		}
		qDebug() << "mplayer" << arguments;
		download = new QProcess(0);
		download->start("mplayer", arguments);
		if (!download->waitForStarted()) {
			errorMsg = notStartError.arg("mplayer");
			return false;
		}
		if (saveOnly) return true;
		arguments.clear();
		arguments.append("-");
	}
	arguments.append("-cache");
	arguments.append("512");
	if (!saveOnly && !saveFileName.isEmpty()) {
		#ifndef QT_4_1_COMPATIBLE
		tee = new QProcess(0);
		QStringList teeArgs;
		teeArgs.append(saveFileName);
		tee->setStandardInputFile(pipeFile); // QT 4.2 function
		player = new QProcess(0);
		tee->setStandardOutputProcess(player); // QT 4.2 function
		qDebug() << "tee" << teeArgs;
		qDebug() << "mplayer" << arguments;
		tee->start("tee", teeArgs);
		if (!tee->waitForStarted()) {
			errorMsg = notStartError.arg("tee");
			return false;
		}
		player->start("mplayer", arguments); // _g_mplayer doesn't like - (/dev/stdin)
		if (!player->waitForStarted()) {
			errorMsg = notStartError.arg("mplayer");
			return false;
		}

		#else
		shell = new QProcess(0);
		QStringList shellArgs;
		shellArgs << "-c" << "tee " + saveFileName + " < " + pipeFile + " | mplayer " + arguments.join(" ");
		qDebug() << "/bin/sh" << shellArgs;
		shell->start("/bin/sh", shellArgs);
		if (!shell->waitForStarted()) {
			errorMsg = notStartError.arg("/bin/sh");
			return false;
		}
		#endif

		return true;
	}
	qDebug() << "gmplayer" << arguments;
	QProcess player(0);
	if (!player.startDetached("gmplayer", arguments)) {
		errorMsg = notStartError.arg("gmplayer");
		return false;
	}
	return true;
}



// Functions
// =========

void initConfigInfo(ConfigInfo& configInfo) {
	configInfo.userConfigFile = QDir::homePath() + "/.teleschorschrc";
	configInfo.systemConfigFile = "/etc/teleschorschrc";

	// User config file
	if (!configInfo.userConfigFile.isEmpty()) {
		QFile configFileUser(configInfo.userConfigFile);
		if (configFileUser.exists()) {configInfo.usedConfigFile  = configInfo.userConfigFile;}
	}

	// System config file
	if (configInfo.usedConfigFile.isEmpty()) {
		QFile configFileSystem(configInfo.systemConfigFile);
		if (configFileSystem.exists()) configInfo.usedConfigFile = configInfo.systemConfigFile;
	}
}


bool addConfig(const QString& fileName, ChannelVec& cv, QString& error) {
	QFile file(fileName);
	if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {error = QObject::tr("Could not open file %1").arg(fileName); return false;}
	Channel channel; // current channel
	bool sectionOccured = false;
	while (!file.atEnd()) {
		QString line = QString(file.readLine()).trimmed();
		if (line.startsWith("#") || line.startsWith(";") || line.isEmpty()) continue; // strip comments
		
		// Is the line a [section]?
		bool section = line.startsWith("[") && line.endsWith("]") && line.size() >= 3;
		if (section) {
			if (sectionOccured) cv.push_back(channel);
			sectionOccured = true;
			channel = Channel();
			channel.name = line.mid(1, line.size()-2);
		}

		// Read properties
		int eqPos = line.indexOf("=");
		bool property = !section && eqPos >= 1;
		if (property) {
			QString propKey = line.left(eqPos);
			QString propVal = line.right(line.size()-eqPos-1);
			if (!sectionOccured) {error = QObject::tr("Property %1 is only allowed in a [section].").arg(propKey); return false;}

			if (propKey == "FULLNAME") channel.fullName = propVal;
			else if (propKey == "STATICURL") channel.staticUrl = propVal;
			else if (propKey == "PLAYER") channel.player = propVal;
			else {error = QObject::tr("Unknown key in ini file: %1").arg(propKey); return false;}
		}

		if (!section && !property) {error = QObject::tr("Line %1 is not valid.").arg(line); return false;}
	}
	if (sectionOccured) cv.push_back(channel);
	return true;
}


QString readChannelVec(const ConfigInfo& configInfo, ChannelVec& channelVec) {
	if (configInfo.usedConfigFile.isEmpty()) return QObject::tr("Neither %1 nor %2 found.").arg(configInfo.systemConfigFile).arg(configInfo.userConfigFile);
	QString error;
	if (!addConfig(configInfo.usedConfigFile, channelVec, error)) return QObject::tr("Error: %1.").arg(error);
	return error;
}


/// \brief Finds a value for a specified variable and appends it to a string.
///
/// \param[in]  var    variable to substitute. These are the possibilities:
///                    - d  day of month (01-31)
///                    - m  month (01-12)
///                    - y  year (last two digits)
///                    - Y  year (4 digits)
///                    - dow_DE  day of week in German (Montag, ...)
/// \param[in]  date   date that should be used when replacing the date dependend variables
/// \param[out] result The determined value of the variable is _appended_.
/// \param[out] error  error message in error cases
/// \returns true in case of success.
bool substituteVar(const QString& var, QDate date, QString& result, QString& errorMsg) {
	if (var == "d") {result += date.toString("dd"); return true;}
	if (var == "m") {result += date.toString("MM"); return true;}
	if (var == "y") {result += date.toString("yy"); return true;}
	if (var == "Y") {result += date.toString("yyyy"); return true;}
	if (var == "dow_DE") {
		int dow = date.dayOfWeek() - 1;
		static const char dow_de[][16] = {"Montag", "Dienstag", "Mittwoch", "Donnerstag", "Freitag", "Samstag", "Sonntag"};
		if (dow >= 0 && dow < 7) {result += dow_de[dow]; return true;}
	}
	errorMsg = QObject::tr("No match for variable %1").arg(var);
	return false;
}


/// \brief Finds a string, where variables of the form <code>${var}</code> are substituted in input (normally a staticUrl).
///
/// - ${d}  day of month (01-31)
/// - ${m}  month (01-12)
/// - ${y}  year (last two digits)
/// - ${Y}  year (4 digits)
/// - ${dow_DE}  day of week in German (Montag, ...)
///
/// \param[in]  input  string where the substitution should be done.
/// \param[in]  date   date that should be used when replacing the date dependend variables
/// \param[out] result The result string with the variables substituted is appended here
/// \param[out] error  error message in error cases
/// \returns true in case of success.
bool substituteVars(const QString& input, QDate date, QString& result, QString& errorMsg) {
	int pos = 0;
	int lastPos = 0;
	while (pos != -1) {
		pos = input.indexOf("${", lastPos);
		if (pos == -1) result += input.mid(lastPos);
		else result += input.mid(lastPos, pos-lastPos);
		lastPos = pos;
		
		// Match?
		if (pos != -1) {
			pos = input.indexOf("}", lastPos);
			if (pos == -1) {
				errorMsg = QObject::tr("${ not closed with }.");
				return false;
			}
			QString var = input.mid(lastPos+2, pos-lastPos-2);
			if (!substituteVar(var, date, result, errorMsg)) return false;
			lastPos = pos+1;
		}
	}
	return true;
}


/// \brief Calls the /bin/sh shell with the specified command and appends the result to a string variable.
///
/// \param[in]  command command to execute with <code>/bin/sh -c "command"</code>
/// \param[out] result  The result string where the output is appended at.
/// \param[out] error   error message in error cases
/// \returns true in case of success.
bool executeShellCommand(const QString& command, QString& result, QString& errorMsg) {
	QProcess evalUrl(0);
	QString cmd = "/bin/sh -c \"" + command + "\"";
	evalUrl.start(cmd);
	if (evalUrl.waitForFinished(3000)) {
		QByteArray newResult = evalUrl.readAllStandardOutput();
		if (result != newResult) {result += newResult.trimmed();}
		return true;
	} 
	errorMsg = QObject::tr("Shell command executed when substituting URL (%1) timed out.").arg(command);
	return false;
}


/// \brief Evaluates the staticUrl
///
/// - Variables of the form ${var} are substituted according the the function ::substituteVars
/// - If the staticUrl is enclosed by backticks it is evaluated by a shell command (after the substitution mentioned above)
///
/// \param[in]  staticUrl string where the substitution should be done.
/// \param[in]  date      date that should be used when replacing the date dependend variables
/// \param[out] result    The result string is appended here
/// \param[out] error     error message in error cases
/// \returns true in case of success.
bool evaluateStaticUrl(const QString& staticUrl, QDate date, QString& result, QString& errorMsg) {
	QString subst;
	bool success = substituteVars(staticUrl, date, subst, errorMsg);
	if (!success) return false;

	// Evaluate staticUrl - it might be dynamic despite its name :-)
	QString cmdres;
	if (subst.left(1) == "`" && subst.right(1) == "`") {
		success = executeShellCommand(subst.mid(1, subst.size()-2), cmdres, errorMsg);
		if (!success) return false;
		result = cmdres;
	} else result = subst;
	return true;
}


void showHelp() {
	// Isn't there a way to output QStrings directly?
	std::cout << QObject::tr("Usage: qteleschorsch [options]").toStdString() << std::endl
	<< QObject::tr("Options: --help: Shows this message at the command line and exits qteleschorsch").toStdString() << std::endl
	<< QObject::tr("More help: man qteleschorsch").toStdString() << std::endl;
}



// MainDialog
// ==========

MainDialog::MainDialog(QWidget *parent): QDialog(parent) {
	player = 0;
	
	// User interface
	setupUi(this);
	QObject::connect(btnOptions, SIGNAL(clicked()), this, SLOT(editOptions()));
	QObject::connect(btnStart, SIGNAL(clicked()), this, SLOT(startAction()));
	QObject::connect(btnStartSave, SIGNAL(clicked()), this, SLOT(startSaveAction()));
	QObject::connect(btnSave, SIGNAL(clicked()), this, SLOT(saveAction()));

	// Init configInfo
	initConfigInfo(configInfo);

	// Read config
	QString error = readChannelVec(configInfo, channelVec);
	if (!error.isEmpty()) QMessageBox::warning(this, tr("Problem when reading the configuration file"), error);

	// Fill in channels
	updateLwChannels();

	// Default date
	QDateTime dateTime = QDateTime::currentDateTime(); // set the default date to today if it is past midday
	if (dateTime.time().hour() < 12) dateTime = dateTime.addDays(-1);
	#ifndef QT_4_1_COMPATIBLE
	calDate->setSelectedDate(dateTime.date());
	#else
	calDateEdit->setDate(dateTime.date());
	#endif
}


MainDialog::~MainDialog() {
	delete player;
}


bool MainDialog::play(bool play, bool save) {
	// Read data from form
	int row = lwChannels->currentRow();
	if (row < 0) return false;

	Channel channel = channelVec[row];
	#ifndef QT_4_1_COMPATIBLE
	QDate date = calDate->selectedDate();
	#else
	QDate date = calDateEdit->date();
	#endif
	// Substitude URL
	QString substUrl;
	QString errorMsg;
	if (!evaluateStaticUrl(channel.staticUrl, date, substUrl, errorMsg)) {
		QMessageBox::warning(this, tr("Problem when substituting URL"), errorMsg);
		return false;
	}

	// Determine player to use
	delete player; 
	player = 0;
	if (channel.player.contains("vlc")) player = new VlcPlayer();
	if (channel.player.contains("mplayer")) player = new MPlayer();
	if (!player) {
		QMessageBox::warning(this, tr("Unknown player"), tr("player %1 not known").arg(channel.player));
		return false;
	}

	// Ask filename
	QString fileName;
	if (save) {
		fileName = QFileDialog::getSaveFileName(this, tr("Save as ..."), QDir::homePath() + "/" + channel.name + "_" + date.toString(Qt::ISODate) +  (channel.player.contains("vlc") ? ".mpeg" : ""), (channel.player.contains("vlc") ? tr("Videos (*.mpeg)") : tr("Videos (*.*)")));
		if (fileName.isEmpty()) return true;
	}

	// Play or Save
	if (!player->play(substUrl, teOffset->time(), !play, fileName, errorMsg)) {
		QMessageBox::warning(this, tr("Error while playing"), errorMsg);
		return false;
	}
	return true;
}



void MainDialog::editOptions() {
	OptionsDialog *od = new OptionsDialog();
	if (od->exec(configInfo.userConfigFile)) {
		channelVec.clear();
		QString error = readChannelVec(configInfo, channelVec);
		if (!error.isEmpty()) QMessageBox::warning(this, tr("Problem when reading the configuration file"), error);
		updateLwChannels();
	}
	delete od;
}


void MainDialog::updateLwChannels() {
	int row = lwChannels->currentRow(); // remember selected row
	lwChannels->clear();
	for (int i = 0; i != channelVec.size(); ++i) lwChannels->addItem(channelVec[i].fullName);
	if (row > lwChannels->count()) row = lwChannels->count()-1; // set to last row. if count==0, row gets -1.
	if (row == -1 && lwChannels->count() > 0) row = 0; // set to first if nothing was selected previously.
	if (row != -1) lwChannels->setCurrentRow(row);
}


bool MainDialog::startAction() {
		return play(true, false);
}


bool MainDialog::startSaveAction() {
		return play(true, true);
}


bool MainDialog::saveAction() {
		return play(false, true);
}



// Main function
// =============

int main(int argc, char *argv[]) {
	QApplication app(argc, argv);

	// Initialize translation
	QString locale = QLocale::system().name();
	QTranslator translator;
	translator.load(QString(":/qteleschorsch_") + locale);
	app.installTranslator(&translator);

	// Parse command line options
	if (app.arguments().size() > 1) {
		showHelp();
		return 0;
	}

	MainDialog *mainDialog = new MainDialog();
	mainDialog->setWindowFlags(Qt::Window);
	mainDialog->show();
	int status = app.exec();
	delete mainDialog;
	return status;
} 
