/** \mainpage Teleschorsch
 *
 * Created by <a href="http://www.toastfreeware.priv.at/">toastfreeware</a>.
 */

#ifndef MAIN_H
#define MAIN_H
#include <QtCore>
#ifndef QT_4_1_COMPATIBLE
#include "ui_maindialog.h"
#else
#include "ui_maindialog_4.1.h"
#endif

// Types
// =====

struct Channel {
	QString name;
	QString fullName;
	QString staticUrl;
	QString player;
};


typedef QVector<Channel> ChannelVec;


/// Contains information about the configuration file locations
struct ConfigInfo {
	QString home;               ///< environment variable $HOME (e.g. /home/gregoa) or empty if $HOME does not exist
	QString systemConfigFile;   ///< system configuration file name (/etc/teleschorschrc)
	QString userConfigFile;     ///< user configuration file name (e.g. /home/gregoa/.teleschorschrc)
	QString usedConfigFile;     ///< currently used config file (e.g. /home/gregoa/.teleschorschrc)
};


/// Represents a media player, e.g. VLC.
struct Player {
	/// Name of the player, e.g. VLC.
	QString name;

	/// \brief (Optionally) play and (optionally) save an URL.
	/// \param[in] urlToPlay specifies what should be played or saved.
	/// \param[in] offset where to start playing in the stream
	/// \param[in] saveOnly if the URL should _not_ be played (used to only save it).
	/// \param[in] saveFileName If this parameter is not empty, the file is (additionally) saved to the local file.
	/// \param[out] errorMsg if an error occurs, this parameter contains a (hopefully :-) ) easy to understand
	///             error message in the current language
	/// \return true if the run was successful. If saveOnly is true and saveFileName is empty, nothing is done and true is returned.
	virtual bool play(QString urlToPlay, QTime offset, bool saveOnly, QString saveFileName, QString& errorMsg) = 0;
	virtual ~Player() {}
};


struct VlcPlayer: public Player {
	VlcPlayer();
	virtual bool play(QString urlToPlay, QTime offset, bool saveOnly, QString saveFileName, QString& errorMsg);
};


struct MPlayer: public Player {
	MPlayer();
	~MPlayer();
	QProcess* download;
	#ifndef QT_4_1_COMPATIBLE
	QProcess* tee;
	QProcess* player;
	#else
	QProcess* shell;
	#endif
	QString tempFile;
	void stopProcesses();
	virtual bool play(QString urlToPlay, QTime offset, bool saveOnly, QString saveFileName, QString& errorMsg);
};



// Functions
// =========

/// Determine ConfigInfo file names
void initConfigInfo(ConfigInfo& configInfo);


/// Adds the content of the specified config file to the channel vector
bool addConfig(const QString& fileName, ChannelVec& cv, QString& error);


/// \brief Read channel vector
///
/// In an error case, the error message is returned.
QString readChannelVec(const ConfigInfo& configInfo, ChannelVec& channelVec);



// MainDialog
// ==========

/// Represents the main dialog.
class MainDialog: public QDialog, private Ui::MainDialog {
Q_OBJECT

private:
	ChannelVec channelVec;
	ConfigInfo configInfo;
	Player* player;

public:
	/// Constructor
	MainDialog(QWidget *parent = 0);
	virtual ~MainDialog();

private:
	/// \brief Starts(plays) and/or saves the selected stream.
	///
	/// It is used to be called by startAction(), startSaveAction() and saveAction().
	/// \return false if player did not start successfully (true if the user aborts).
	bool play(bool play, bool save);

public slots:
	/// Shows a dialog where the user can edit the config file.
	void editOptions();

	/// Updates the "QListWidget" in the user interface that shows the cannel list.
	void updateLwChannels();

	/// \brief Starts (plays) the selected stream.
	/// \return true if player starts successfully.
	bool startAction();

	/// \brief Starts (plays) and saves the selected stream.
	/// \return true if player starts successfully.
	bool startSaveAction();

	/// \brief Saves the selected stream.
	/// \return true if player starts successfully - the current
	/// implementation lets the player save the stream.
	bool saveAction();
};


void appendPlayerOptions(const QString& player, QTime offset, QStringList& arguments);


#endif
