#!/bin/bash

# Copyright and License:
#
# Copyright (C) 2007-2009
# gregor herrmann <gregor+debian@comodo.priv.at>,
# Philipp Spitzer <philipp@spitzer.priv.at>
#
# This program is free software; you can redistribute it and/or modify it   
# under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA, or point
# your browser to http://www.gnu.org/licenses/gpl.html

# Changes:
#
# 0.1.1
# 	add command line options "-xdialog" (default) and "-dialog"
#
# 0.1.0
# 	use the new configuration file
#
# 0.0.6
# 	move mplayer start options down
#
# 0.0.5
# 	implement some more error checks
#
# 0.0.4
# 	set default date to yesterday if current time < noon
#
# 0.0.3:
#		clarify description of "start time", i.e. "offset"
#
# 0.0.2:
# 	check for required binaries
#
# 0.0.1:
# 	first release

# functions

error() {
	echo "$(basename $0) aborting with error:"
	echo $1
	exit 1
}

# parameters

UI="xdialog"
if [ -n "$1" ] ; then
	case "$1" in
		"-xdialog")
			UI="xdialog"
			;;
		"-dialog")
			UI="dialog"
			;;
	esac
fi

# checks

case "$UI" in
	"xdialog")
		which Xdialog > /dev/null || error "Please install Xdialog or try -dialog."
		;;
	"dialog")
		which dialog > /dev/null || error "Please install dialog or try -xdialog."
		;;
esac		

which vlc > /dev/null || error "Please install vlc."
which gmplayer > /dev/null || error "Please install (g)mplayer."

# some variables

if [ -r $HOME/.teleschorschrc ] ; then
	CONFIG=$HOME/.teleschorschrc
elif [ -r /etc/teleschorschrc ] ; then
	CONFIG=/etc/teleschorschrc
else
	error "Neiter /etc/teleschorschrc nor $HOME/.teleschorschrc found"
fi

if [ $(date +"%H") -lt 12 ] ; then
	DEFAULT_DATE=$(date --date "yesterday" +"%d %m %Y")
else
	DEFAULT_DATE=$(date +"%d %m %Y")
fi

OUTFILE="/tmp/$(basename $0).tmp.$$"

# start the fun

# read config

while read LINE ; do
	if [ -z "$LINE" ] ; then continue; fi
	if echo $LINE | egrep "^[#;]" > /dev/null ; then continue; fi
	if echo $LINE | egrep "^\[" > /dev/null; then
		if [ -n "$C" ] ; then CHANNELS+=("$C") ; fi
		C="$LINE"
	else
		C+="~$LINE"
	fi
done < $CONFIG
CHANNELS+=("$C")

COUNT="${#CHANNELS[*]}"

# create channel list for (x)dialog

for ((i=0; i < COUNT; i++)) ; do # bashism
	LIST+=$(echo ${CHANNELS[$i]} | sed -e "s/\[\(.\+\)\]/$i/" -e 's/FULLNAME=//' -e 's/ /\xa0/g' | awk -F '~' '{printf "%s\t%s\toff ", $1, $2}')
done

# call (x)dialog
case "$UI" in
	"xdialog")
		Xdialog --title "TeleSchorsch" \
			--no-tags \
			--radiolist "Choose your preferred stream:" 0 0 0 \
			$LIST \
			--calendar "Date" 0 0 $DEFAULT_DATE \
			--timebox "Offset\n(Start at beginning of stream: 0:0:0, start at position 3 minutes: 0:3:0, ...)" 0 0 0 0 0 \
			2>$OUTFILE
		;;
	"dialog")
		dialog --title "TeleSchorsch" \
		 --radiolist "Choose your preferred stream" 0 0 0 \
			$LIST \
			--calendar "Date" 0 0 $DEFAULT_DATE \
			--timebox "Offset\n(Start at beginning of stream: 0:0:0, start at position 3 minutes: 0:3:0, ...)" 0 0 0 0 0 \
			2>$OUTFILE
		;;
esac

RETVAL=$?

case $RETVAL in
	0)
		;;
	*)
		echo "$(basename $0) exiting (user abort, error, whatnot) with return value: $RETVAL"
		exit $RETVAL
		;;
esac

[ -e "$OUTFILE" ] && perl -pi -e 's;\n; ;' $OUTFILE \
	|| error "$OUTFILE not found"
read STREAM DATE OFFSET < $OUTFILE
[ -n "$STREAM" ] && [ -n "$DATE" ] && [ -n "$OFFSET" ] && rm -f $OUTFILE \
	|| error "Problem with variables, check contents of $OUTFILE"

DATE=$(echo $DATE | perl -p -e 's;(.+)/(.+)/(.+);$3-$2-$1;')
SECONDS=$(echo $OFFSET | perl -n -e '($h,$m,$s) = split(/:/); print $h*3600 + $m*60 + $s;')
y=$(date --date "$DATE" +"%y")
Y=$(date --date "$DATE" +"%Y")
m=$(date --date "$DATE" +"%m")
d=$(date --date "$DATE" +"%d")
dow_DE=$(LANG=$(locale -a | grep ^de_ | head -n 1) date --date "$DATE" +"%A")

URL=$(echo "${CHANNELS[$STREAM]}" | perl -pe 's/^.*STATICURL=(.+?)(~.*|$)/$1/')
PLAYER=$(echo "${CHANNELS[$STREAM]}" | perl -pe 's/^.*PLAYER=(.+?)(~.*|$)/$1/')

case $PLAYER in
	gmplayer)
		STARTOPTION="-cache 512 -ss"
		;;
	vlc)
		STARTOPTION="--start-time"
		;;
	*)
		error "Unknown PLAYER: $PLAYER"
		;;
esac

$PLAYER $(eval echo $URL) $STARTOPTION $SECONDS
